#ifndef switchDebounce_h
#define switchDebounce_h
/*
 DebouncedSwitch debounces mechanical swithes connected to INPUT_PULLUP digital inputs
 DebouncedSwitch adds about ?? bytes to the program and uses about 3 bytes RAM per switch

 (c)2012 Forward Computing and Control Pty. Ltd. 
 This code may be freely used for both private and commerical use.
 Provide this copyright is maintained.
 */

#include "Arduino.h"


class DebouncedSwitch {
public:	
DebouncedSwitch(int pin);	
void update(void);
bool isSwitch(unsigned char state);

static const unsigned char SwitchChanged = 0;
static const unsigned char SwitchDown = 1;
static const unsigned char SwitchClicked = 2;
static const unsigned char SwitchPressed = 3;
static const unsigned char SwitchDownRepeat = 4;
static const unsigned char SwitchUpTimeout = 5;

private:
int switchPin;
long lastMillis;
struct _reg_SwitchStatus {
	unsigned char Changed : 1;	// set if debounced switch state changed from up / down, set for one loop only
	unsigned char Down : 1;	// set if switch down, closed, digital input connected to GND
        unsigned char Click : 1; // set if switch pressed and released in short (click) periods, set on release. set for one loop only
	unsigned char Pressed : 1;	// set if switch pressed for more then Press_count, i.e. long press, set for one loop only and not set again until switch released and pressed again.
	unsigned char DownRepeat : 1; // set repeatedly if switch pressed for more then RepeatedPress_Count, each set for one loop only, reset again after RepeatedPress_Count
	// cleared on next loop and counter rest for repeated firing
	unsigned char UpTimeout : 1;	// set switch has been released for more then the Release_Count, set for one loop only, reset again after Release_Count
	// cleared on next loop and counter rest for repeated firing
	unsigned char lastSwitchReading : 1; // set if switch down (GND) on prevous update not debounced
	unsigned char SwitchStatus_7 : 1;	//
};


union _reg_SwitchStatus_Union {
	byte _reg;
	struct _reg_SwitchStatus _struct;
};

/// Switch Debounce 
byte Debounce_Count_Low;  // 50 = about 0.05 sec  = 50 x 1mS
byte Debounce_Count_High; // 50 = about 0.05 sec  = 50 x 1mS
byte SwitchStatus;  // hold the 8 status bits
byte debounceCounter;
};

#endif // switchDebounce_h
	
DebouncedSwitch sw(4);

/*
  Blink
  Turns on an LED on for one second, then off for one second, repeatedly.

  This example code is in the public domain.
 */

// Pin 13 has an LED connected on most Arduino boards.
// give it a name:
int led = 13;

// the setup routine runs once when you press reset:
void setup() {
  // initialize the digital pin as an output.
  pinMode(led, OUTPUT);
  Serial.begin(9600);
}

// the loop routine runs over and over again forever:
void loop() {
  sw.update();
  if (sw.isSwitch(sw.SwitchDown)) {
  digitalWrite(led, HIGH);   // turn the LED on (HIGH is the voltage level)
  } else {
  digitalWrite(led, LOW);    // turn the LED off by making the voltage LOW
  }
}



//#include "DebouncedSwitch.h"
bool DebouncedSwitch::isSwitch(unsigned char state) {
    return SwitchStatus & (1 << state); //  == _BV bit value macro
}

DebouncedSwitch::DebouncedSwitch(int pin) {
	switchPin = pin;
	pinMode(switchPin, INPUT_PULLUP);
	Debounce_Count_Low = 50;
	Debounce_Count_High = 50;
	SwitchStatus = 0;

	debounceCounter = (Debounce_Count_Low+Debounce_Count_High);
	// set past debouce count so do not fire another switch changed

	// clear the switch flags
	//unsigned char switchPort = SWITCH_IN;
	// is switch is still down here fire switch trigger
	bool val = (digitalRead(switchPin) == HIGH);
	if (!val) {
		// update switch down and changed
		// set last input and set counter
		union _reg_SwitchStatus_Union switchUnion;
		switchUnion._reg = SwitchStatus;
		switchUnion._struct.lastSwitchReading = 1; // 1 for close
		// note this is opposite to pin value
		//  1 on pin => 0 for switch Down and vis versa
		switchUnion._struct.Down = 1;  // note this is opposite to pin value
		//  1 on pin => 0 for switch Down and vis versa
		switchUnion._struct.Changed = 1; // debounced switch changed
		SwitchStatus = switchUnion._reg;
	} else {
		union _reg_SwitchStatus_Union switchUnion;
		switchUnion._reg = SwitchStatus;
		switchUnion._struct.lastSwitchReading = 0; // 1 for close
		// note this is opposite to pin value
		//  1 on pin => 0 for switch Down and vis versa
		switchUnion._struct.Down = 0;  // note this is opposite to pin value
		//  1 on pin => 0 for switch Down and vis versa
		SwitchStatus = switchUnion._reg;
          }  
         lastMillis = millis();
}




// ------------------------
// READ_SWITCH
//  this is called each 1mS
//  checks for debounce and updates
//  SWITCH_FLAGS
//  SwitchDown and SwitchChanged
//
//  SwitchChanged set to 1 each time switch changes state
//  is only set to 1 for one loop cleared each time this is called
//  SwitchDown set to 1 when switch is pressed (debouced) else 0 when up
//
// also the internal flag
//  SwitchLastInput is set to 1 when last switch read was low/on else set 0 for high/off
// ------------------------
void DebouncedSwitch::update(void) {
	long ms = millis(); // only call this once

//    Serial.print(" ------ called Debounce "); Serial.print("ms ");Serial.println(ms);
	long deltaMillis = ms-lastMillis;
//Serial.print("lastMillis ");Serial.println(lastMillis);Serial.print("deltaMillis ");Serial.println(deltaMillis);

	lastMillis = ms;
	// clear debouced switch changed and press timed out  every loop
	union _reg_SwitchStatus_Union switchUnion;
	//unsigned char debounceCounter = debounceCounter; // byte storage
	switchUnion._reg = SwitchStatus;
	switchUnion._struct.Changed = 0;
	switchUnion._struct.Pressed = 0;
	switchUnion._struct.DownRepeat = 0;
	switchUnion._struct.UpTimeout = 0;
	SwitchStatus = switchUnion._reg ;
	
	if (deltaMillis == 0) { // no ticks
		return;
	}
	// this code assumes we are called at least once each mS.
	// if not then the debounce will be a bit slower
  //  Serial.print(" debounceCounter:");   Serial.print(debounceCounter);   Serial.print(" deltaMillis:");   Serial.print(deltaMillis);   Serial.println();
	long dbDelta = deltaMillis + debounceCounter;
	byte lastDebounceCounter = debounceCounter; // for switchChanged
 //   Serial.print(" debounceCounter:");    Serial.print(debounceCounter);   Serial.print(" dbMillis:");   Serial.print(dbDelta);   Serial.println();

	// limit to byte storage
	if (dbDelta >250) {
		debounceCounter = 250;
	} else {
		debounceCounter = dbDelta;
	}
 //   Serial.print(" debounceCounter after update:");   Serial.print(debounceCounter);   Serial.println();
	
	bool val = (digitalRead(switchPin) == HIGH);
//Serial.print("read pin:"); Serial.println(val);
	if (!val) {
		// switch low SW_LOW_PRESSED
//Serial.print(" low SW_LOW_PRESSED"); Serial.println();
		if (switchUnion._struct.lastSwitchReading == 0) {
			// else this sw low and last high, set last input to high and reset debounce counter
			switchUnion._struct.lastSwitchReading = 1;
			debounceCounter = 0;
Serial.print("ms start low "); Serial.println(ms);
//Serial.print(" lastSwitchReding 0 set debouce to 0"); Serial.println();

		} else {
//Serial.print(" low SW_LOW_PRESSED count debug"); Serial.println();  
			// if last input low and this one low check for debounce low
			if ((debounceCounter >= Debounce_Count_Low) && (lastDebounceCounter < Debounce_Count_Low)) {
				switchUnion._struct.Down = 1; //de-bounced
				switchUnion._struct.Changed = 1; //de-bounced switch changed
Serial.print("ms change to low "); Serial.println(ms);

			} // else either waiting for debouce or passed it
		}
	} else {
		// read SW high/released
//Serial.print(" high/released"); Serial.println();
		if (switchUnion._struct.lastSwitchReading == 1) {
			// else this sw low and last high, set last input to high and reset debounce counter
			switchUnion._struct.lastSwitchReading = 0;
			debounceCounter = 0;
Serial.print("ms start high ");Serial.println(ms);
//Serial.print(" lastSwitchReding 1 set debouce to 0"); Serial.println();

		} else {
//Serial.print(" high/released count debug"); Serial.println();  
			// if last input low and this one low check for debounce low
			if ((debounceCounter >= Debounce_Count_High) && (lastDebounceCounter < Debounce_Count_High)) {
				switchUnion._struct.Down = 0; //de-bounced
				switchUnion._struct.Changed = 1; //de-bounced switch changed
Serial.print("ms change to high ");Serial.println(ms);

			} // else either waiting for debounce or passed it
		}
	}
	SwitchStatus = switchUnion._reg ;
//Serial.print(" switch Down is "); Serial.println(isSwitch(SwitchDown));

}

