/**
   webPages.cpp
   by Matthew Ford,  2021/12/06
   (c)2021 Forward Computing and Control Pty. Ltd.
   NSW, Australia  www.forward.com.au
   This code may be freely used for both private and commerical use.
   Provide this copyright is maintained.

*/

// perhaps add disabled to buttons while isBlanking() is true;
#include "webPages.h"
#include "LittleFSsupport.h"
#include "ESP32sntpSupport.h"
#include "DebugOut.h"

#include "tzPosix.h"
//#include "PrintTimes.h"
#include "TZ_support.h"
#include <NetworkClient.h>
#include "ESPAsyncWebServer.h"
#include "switchSettings.h"

extern bool ac240V_active;  // set in EV_Charger_V2H_switch.ino

// normally DEBUG is commented out
//#define DEBUG
static Stream *debugPtr = NULL;  // local to this file

static String processor(const String &var);

static bool convertHH_MMtoMins(SafeString &hh_mm, int &mins_out);

static bool webServerStarted = false;
static AsyncWebServer server(80);

static void print2digits(String &result, uint num);
static String minToHH_mm(int mins_in);

static void handle_root(AsyncWebServerRequest *request);
static void handle_chargerPower(AsyncWebServerRequest *request);
static void handle_chargerAuto(AsyncWebServerRequest *request);
static void handle_v2hPower(AsyncWebServerRequest *request);
static void handle_v2hAuto(AsyncWebServerRequest *request);
static void handle_getState(AsyncWebServerRequest *request);

static void handleSetTimeOnOffPage(AsyncWebServerRequest *request);
static void handle_setTZstr(AsyncWebServerRequest *request);  // setter button
static void handle_resetTZ(AsyncWebServerRequest *request);
static void handle_setTime(AsyncWebServerRequest *request);
static void handle_setTimeOnOff(AsyncWebServerRequest *request);

static void printRequestArgs(Print *outPtr, AsyncWebServerRequest *request);
static void handleNotFound(AsyncWebServerRequest *request);
static void redirect(AsyncWebServerRequest *request, const char *url);
static void returnOK(AsyncWebServerRequest *request);
static void returnFail(AsyncWebServerRequest *request, String msg);

static String correctedTZstr;  // empty if no problems
static String correctedTZstrDsc;
static String userInputTZstr;
static String setMsg;
static const int DEFAULT_REFRESH = 70;
static int refreshSec = DEFAULT_REFRESH;  // set by power handlers, reset by processor
static bool fastRefreshNeeded = false;    // next /api/state will return 1 seconds


void startWebServer() {
#ifdef DEBUG
  debugPtr = getDebugOut();
#endif
  if (webServerStarted) {
    return;
  }
  if (!initializeFS()) {
    if (debugPtr) {
      debugPtr->println("LittleFS failed to start.");
    }
  }

  server.on("/", HTTP_GET, handle_root);

  server.on("/index.html", HTTP_GET, [](AsyncWebServerRequest *request) {
    if (debugPtr) {
      debugPtr->println("get /index.html");
    }
    AsyncWebServerResponse *response = request->beginResponse(LittleFS, "/index.html", "text/html");
    response->addHeader("Cache-Control", "public, max-age=2592000");  // 30 days cache
    request->send(response);
    if (debugPtr) {
      debugPtr->println("get /index.html  complete");
    }
  });

  server.on("/chargerPower", HTTP_POST, handle_chargerPower);
  server.on("/chargerAuto", HTTP_POST, handle_chargerAuto);
  server.on("/v2hPower", HTTP_POST, handle_v2hPower);
  server.on("/v2hAuto", HTTP_POST, handle_v2hAuto);

  server.on("/api/state", HTTP_GET, handle_getState);

  server.on("/settz.html", HTTP_GET, [](AsyncWebServerRequest *request) {
    request->send(LittleFS, "/settz.html", String(), false, processor);
  });
  server.on("/setTimeOnOff", HTTP_GET, handleSetTimeOnOffPage);
  server.on("/setTimeOnOff", HTTP_POST, handle_setTimeOnOff);

  server.on("/setTZstr", HTTP_POST, handle_setTZstr);
  server.on("/resetTZ", HTTP_GET, handle_resetTZ);
  server.on("/setTime", HTTP_POST, handle_setTime);

  // Route to load style.css file with caching
  server.on("/style.css", HTTP_GET, [](AsyncWebServerRequest *request) {
    if (debugPtr) {
      debugPtr->println("get /style.css");
    }
    AsyncWebServerResponse *response = request->beginResponse(LittleFS, "/style.css", "text/css");
    response->addHeader("Cache-Control", "public, max-age=2592000");  // 30 days cache
    request->send(response);
    if (debugPtr) {
      debugPtr->println("get /style.css  complete");
    }
  });

  // Route to load state.js file with caching
  server.on("/state.js", HTTP_GET, [](AsyncWebServerRequest *request) {
    AsyncWebServerResponse *response = request->beginResponse(LittleFS, "/state.js", "text/javascript");
    response->addHeader("Cache-Control", "public, max-age=2592000");  // 30 days cache
    request->send(response);
  });

  server.onNotFound(handleNotFound);
  (void)(returnOK);    // to suppress compiler warning only
  (void)(returnFail);  // to suppress compiler warning only

  server.begin();
  if (debugPtr) {
    debugPtr->println("Web server started");
  }
  webServerStarted = true;
}



static void redirect(AsyncWebServerRequest *request, const char *url) {
  if (debugPtr) {
    debugPtr->print("Redirect to: ");
    debugPtr->println(url);
  }
  request->redirect(url);
}

static void handle_root(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->println("get /");
  }
  redirect(request, "/index.html");
  if (debugPtr) {
    debugPtr->println("get / complete");
  }
}

static void returnOK(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->print("Return OK (empty plain text)");
    debugPtr->println();
  }
  request->send(200, "text/plain", "");
}

static void returnFail(AsyncWebServerRequest *request, String msg) {
  msg += "\r\n";
  if (debugPtr) {
    debugPtr->print("Return Fail with msg: ");
    debugPtr->println(msg);
  }
  request->send(500, "text/plain", msg);
}

static void printRequestArgs(Print *outPtr, AsyncWebServerRequest *request) {
  if (!outPtr) {
    return;
  }
  outPtr->print("URI: ");
  outPtr->print(request->url());
  outPtr->print("   Method: ");
  outPtr->println((request->method() == HTTP_GET) ? "GET" : "POST");
  outPtr->print(" Arguments: ");
  outPtr->println(request->params());
  for (int i = 0; i < request->params(); i++) {
    const AsyncWebParameter *p = request->getParam(i);
    outPtr->print(" NAME:");
    outPtr->print(p->name());
    outPtr->print("   VALUE:");
    outPtr->println(p->value());
  }
}

static void handleNotFound(AsyncWebServerRequest *request) {
  String message = "Not found: ";
  message += (request->method() == HTTP_GET) ? "GET " : "POST ";
  message += request->url();
  request->send(404, "text/plain", message);
}



void resetTZ() {
  resetDefaultTZstr();
  saveTZconfigIfNeeded();  // write to file
}


// return true if valid input, result returned in mins_out var
// convert to mins
static bool convertHH_MMtoMins(SafeString &hh_mm, int &mins_out) {
  cSF(token, 20);
  int idx = 0;
  idx = hh_mm.stoken(token, idx, ':');
  int hr = 0;
  int mins = 0;
  if (token.toInt(hr) && (hr >= 0) && (hr <= 23)) {
    idx = hh_mm.stoken(token, idx, ':');
    if (token.toInt(mins) && (mins >= 0) && (mins <= 59)) {
      mins_out = hr * 60 + mins;
      if (debugPtr) {
        debugPtr->print("dayMins:");
        debugPtr->print(mins_out);
        debugPtr->println();
      }
      return true;
    } else {
      return false;
    }
  } else {
    return false;
  }
}

// GET handler for /setTimeOnOff - serves the page with device-specific times
static void handleSetTimeOnOffPage(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->print(" handleSetTimeOnOffPage (GET) ");
    debugPtr->println();
    printRequestArgs(debugPtr, request);
  }

  // Get device parameter to determine which device's times to show
  String device = "";
  if (request->hasParam("device")) {
    device = request->getParam("device")->value();
  }
  if (debugPtr) {
    debugPtr->printf("Device: %s\n", device.c_str());
  }

  // Create a processor function that captures device context
  using ProcessorFunc = std::function<String(const String &)>;
  ProcessorFunc deviceProcessor = [device](const String &var) -> String {
    String rtnString = "";
    if (var == "DEVICE") {
      rtnString = (device == "v2h") ? "v2h" : "charger";
    } else if (var == "DEVICE_NAME") {
      rtnString = (device == "v2h") ? "V2H" : "Charger";
    } else if (var == "TIME_ON") {
      if (device == "v2h") {
        rtnString = minToHH_mm(getV2lOnTime());
      } else {
        // Default to charger
        rtnString = minToHH_mm(getChargerOnTime());
      }
    } else if (var == "TIME_OFF") {
      if (device == "v2h") {
        rtnString = minToHH_mm(getV2lOffTime());
      } else {
        // Default to charger
        rtnString = minToHH_mm(getChargerOffTime());
      }
    } else if (var == "TIME_CURRENT_S") {
      rtnString = String(getLocalTime_s());
    } else if (var == "HAVE_SNTP") {
      rtnString = String(haveSNTP());
    }
    // For other variables, fall through to main processor
    if (rtnString.length() == 0) {
      rtnString = processor(var);
    }
    return rtnString;
  };

  request->send(LittleFS, "/setTimeOnOff.html", String(), false, deviceProcessor);
}

static void handle_setTimeOnOff(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->print(" handle_setTimeOnOff (POST) ");
    debugPtr->println();
    printRequestArgs(debugPtr, request);
  }

  // Handle POST - process form data
  cSF(sfTimeSetting, 20);
  if (debugPtr) {
    debugPtr->print("SetOnOFF Time (POST): ");
  }

  // Get device parameter to determine which device's times to set
  String device = "";
  int onMins = 0;
  int offMins = 0;
  bool onTimeValid = false;
  bool offTimeValid = false;

  // Parse parameters by index to find device, TIME_ON, and TIME_OFF
  for (int i = 0; i < request->params(); i++) {
    const AsyncWebParameter *p = request->getParam(i);
    if (strcmp(p->name().c_str(), "device") == 0) {
      device = p->value();
      if (debugPtr) {
        debugPtr->printf("Device: %s\n", device.c_str());
      }
    } else if (strcmp(p->name().c_str(), "TIME_ON") == 0) {
      if (debugPtr) {
        debugPtr->printf("Time ON Value: %s\n", p->value().c_str());
      }
      sfTimeSetting = p->value().c_str();
      if (convertHH_MMtoMins(sfTimeSetting, onMins)) {
        onTimeValid = true;
      }
    } else if (strcmp(p->name().c_str(), "TIME_OFF") == 0) {
      if (debugPtr) {
        debugPtr->printf("Time OFF Value: %s\n", p->value().c_str());
      }
      sfTimeSetting = p->value().c_str();
      if (convertHH_MMtoMins(sfTimeSetting, offMins)) {
        offTimeValid = true;
      }
    }
  }

  // Set on/off times for the appropriate device
  if (device == "v2h") {
    if (onTimeValid) {
      setV2lOnTime(onMins);
    }
    if (offTimeValid) {
      setV2lOffTime(offMins);
    }
  } else {
    // Default to charger
    if (onTimeValid) {
      setChargerOnTime(onMins);
    }
    if (offTimeValid) {
      setChargerOffTime(offMins);
    }
  }

  redirect(request, "/index.html");
}




// does the %..% replacement
// this processor handles ALL but the setOnOff times
static String processor(const String &var) {
  String rtnString = "";
  if (debugPtr) {
    debugPtr->print("processing %");
    debugPtr->print(var);
    debugPtr->print("% = '");
  }
  if (var == "CHARGER_STATE") {
    rtnString = getChargerIsOn() ? "on" : "";
  } else if (var == "CHARGER_AUTO_ACTIVE") {
    rtnString = getChargerIsAuto() ? "active" : "";
  } else if (var == "CHARGER_TIME_ON") {
    rtnString = minToHH_mm(getChargerOnTime());
  } else if (var == "CHARGER_TIME_OFF") {
    rtnString = minToHH_mm(getChargerOffTime());
  } else if (var == "V2H_STATE") {
    rtnString = getV2lIsOn() ? "on" : "";
  } else if (var == "V2H_AUTO_ACTIVE") {
    rtnString = getV2lIsAuto() ? "active" : "";
  } else if (var == "V2H_TIME_ON") {
    rtnString = minToHH_mm(getV2lOnTime());
  } else if (var == "V2H_TIME_OFF") {
    rtnString = minToHH_mm(getV2lOffTime());
  } else if (var == "AC240V_STATUS") {
    rtnString = ac240V_active ? "active" : "";
  } else if (var == "REFRESH_SECONDS") {
    if (isBlanking() || fastRefreshNeeded || (!haveSNTP())) {  // refresh each 1 until get sntp
        fastRefreshNeeded = false;
        refreshSec = 1;
      }
    else {
      refreshSec = DEFAULT_REFRESH;
    }
    rtnString = refreshSec;
  } else if (var == "TZ_DESC") {
    String TZstr = getTZstr();
    struct posix_tz_data_struct tzdata;
    posixTZDataFromStr(TZstr, tzdata);
    buildPOSIXdescription(tzdata, rtnString);
    rtnString.replace("\n", "<br>");
  } else if (var == "TZ_STR") {
    if (setMsg.length()) {
      rtnString = correctedTZstr;
      correctedTZstr = "";  // finished  with this
    } else {
      rtnString = getTZstr();
    }
  } else if (var == "TZ_DESC_STR") {
    if (correctedTZstrDsc.length()) {
      rtnString = correctedTZstrDsc;
      correctedTZstrDsc = "";  // finished  with this
    } else {
      rtnString = getTZstr();
    }
    String TZstr = rtnString;
    struct posix_tz_data_struct tzdata;
    posixTZDataFromStr(TZstr, tzdata);
    buildPOSIXdescription(tzdata, rtnString);
    rtnString.replace("\n", "<br>");
  } else if (var == "TZ_CORRECTED") {
    if (userInputTZstr.length()) {
      rtnString = "Time Zone string has been cleaned up from<br>";
      rtnString += userInputTZstr;
      //   AEST-10AEDT,M10.1.0,M4.1.0/3
      rtnString += "<br>to";
      userInputTZstr = "";  // finished  with this
    }
  } else if (var == "TZ_STR_SET_CORRECTED") {
    rtnString = setMsg;
    setMsg = "";
  } else if (var == "HAVE_SNTP") {
    rtnString = String(haveSNTP());
  } else if (var == "TZ_VALUE") {
    rtnString = getTZstr();
  } else if (var == "TIME") {
    rtnString = getCurrentTime_hhmm();
  } else if (var == "TIME_CURRENT_S") {  // local time in unix sec
    rtnString = String(getLocalTime_s());
  } else if (var == "TIME_CURRENT_HHMM") {
    rtnString = String(getCurrentTime_hhmm());
  }
  if (debugPtr) {
    debugPtr->print(rtnString);
    debugPtr->println("'");
  }
  return rtnString;
}





static void handle_setTZstr(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->println(" handle_setTZstr");
    printRequestArgs(debugPtr, request);
  }
  String newTZ;
  for (int i = 0; i < request->params(); i++) {
    const AsyncWebParameter *p = request->getParam(i);
    if (strcmp(p->name().c_str(), "TZ_INPUT_STR") == 0) {
      userInputTZstr = "";
      correctedTZstr = "";
      correctedTZstrDsc = "";
      setMsg = "";
      if (debugPtr) {
        debugPtr->printf("Tz input string Value: %s\n", p->value().c_str());
      }
      String inputStr = p->value();
      inputStr.trim();
      String cleanStr = p->value();
      cleanUpPosixTZStr(cleanStr);
      if (inputStr != cleanStr) {
        userInputTZstr = inputStr;
        correctedTZstr = cleanStr;
        correctedTZstrDsc = cleanStr;
        setMsg = "Select the Set TZ String button again to set<br>this cleaned up time zone string";
      } else {
        newTZ = cleanStr;
        correctedTZstrDsc = cleanStr;
      }
    }
  }
  if (setMsg.length()) {
    redirect(request, "/settz.html");
  } else {
    // set new tz
    setTZfromPOSIXstr(newTZ.c_str());  // cleans up and set save flag as well
    redirect(request, "/index.html");
  }
}

static void handle_resetTZ(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->println(" handle_resetTZ");
    printRequestArgs(debugPtr, request);
  }
  resetDefaultTZstr();
  redirect(request, "/index.html");
}

static void handle_setTime(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->println(" handle_setTime");
    printRequestArgs(debugPtr, request);
  }
  cSF(sfTimeSetting, 20);
  if (debugPtr) {
    debugPtr->print("Set time Data: ");
  }
  for (int i = 0; i < request->params(); i++) {
    const AsyncWebParameter *p = request->getParam(i);
    if (strcmp(p->name().c_str(), "TIME") == 0) {
      if (debugPtr) {
        debugPtr->printf("Time Value: %s\n", p->value().c_str());
      }
      sfTimeSetting = p->value().c_str();
      int dayMins = 0;
      if (convertHH_MMtoMins(sfTimeSetting, dayMins)) {
        // get utc time
        time_t now = time(nullptr);
        struct tm *tmPtr = gmtime(&now);
        int utcDayMins = tmPtr->tm_hour * 60 + tmPtr->tm_min;
        int tzDiffMins = (utcDayMins - dayMins);
        if (debugPtr) {
          debugPtr->print("tzDiff :");
          debugPtr->print(tzDiffMins);
          debugPtr->println();
        }
        // round to 5min
        int sign = 1;
        if (tzDiffMins < 0) {
          sign = -1;
        }
        int tzDiffMinRoundedUnsigned = ((tzDiffMins * sign * 60 + 150) / 300 * 300) / 60;
        if (debugPtr) {
          debugPtr->print("tzDiffRounded :");
          debugPtr->print(tzDiffMinRoundedUnsigned);
          debugPtr->println();
        }
        int hr_offset = tzDiffMinRoundedUnsigned / 60;
        int min_offset = tzDiffMinRoundedUnsigned - hr_offset * 60;
        tzDiffMins = tzDiffMinRoundedUnsigned * sign;
        hr_offset *= sign;
        if (debugPtr) {
          debugPtr->print("offset ");
          debugPtr->print(hr_offset);
          debugPtr->print(':');
          debugPtr->print(min_offset);
          debugPtr->println();
        }
        // offsets in range -12 < offset <= +12  i.e. -11:45 is the smallest offset and +12:00 is the largest
        //         mins in range -720 < offsetMin <= +720
        // e.g. LT (localTime) = 14:00,  UTC=04:00  tzoffset = +10:00
        //      LT = 08:00  UTC = 22:00  tzoffset =  -14 => <=-12 so add 24,  -14+24 = +10
        //      LT = 14:00  UTC = 22:00  tzoffset = -8:00
        //      LT = 20:00  UTC = 4:00   tzoffset = 16:00 => >12 so subtract 24,  16-24 = -8:00
        if (tzDiffMins <= -720) {
          tzDiffMins += (24 * 60);
        } else if (tzDiffMins > 720) {
          tzDiffMins -= (24 * 60);
        }
        sign = 1;
        if (tzDiffMins < 0) {
          sign = -1;
        }
        tzDiffMinRoundedUnsigned = tzDiffMins * sign;
        hr_offset = tzDiffMinRoundedUnsigned / 60;
        min_offset = tzDiffMinRoundedUnsigned - hr_offset * 60;
        hr_offset *= sign;
        if (debugPtr) {
          debugPtr->print("offset (+/-12)");
          debugPtr->print(hr_offset);
          debugPtr->print(':');
          debugPtr->print(min_offset);
          debugPtr->println();
        }
        setTZoffsetInMins(tzDiffMins);  // update in min
      }                                 // else error ignore
    }
  }
  redirect(request, "/index.html");
}

// only handles +v numbers
static void print2digits(String &result, uint num) {
  if (num < 10) {
    result += '0';
  }
  result += num;
}

static String minToHH_mm(int mins_in) {
  String rtn;
  int hrs = mins_in / 60;
  int mins = mins_in - (hrs * 60);
  print2digits(rtn, hrs);
  rtn += ':';
  print2digits(rtn, mins);
  return rtn;
}

// POST handler for /chargerPower - Toggle charger power
static void handle_chargerPower(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->println(" handle_chargerPower");
    printRequestArgs(debugPtr, request);
  }
  bool currentState = getChargerIsOn();
  setChargerIsOn(!currentState);  // just set volatiles for later access by loop()
  fastRefreshNeeded = true;       // next /api/state will return 3 seconds
  request->send(200, "application/json", "{\"ok\":true}");
}

// POST handler for /chargerAuto - Toggle charger auto mode
static void handle_chargerAuto(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->println(" handle_chargerAuto");
    printRequestArgs(debugPtr, request);
  }
  bool currentState = getChargerIsAuto();
  setChargerIsAuto(!currentState);
  fastRefreshNeeded = true;  // next /api/state will return 3 seconds
  request->send(200, "application/json", "{\"ok\":true}");
}


// POST handler for /v2hPower - Toggle V2H power
static void handle_v2hPower(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->println(" handle_v2hPower");
    printRequestArgs(debugPtr, request);
  }
  bool currentState = getV2lIsOn();
  setV2lIsOn(!currentState);  // just set volatiles for later access by loop()
  fastRefreshNeeded = true;   // next /api/state will return 3 seconds
  request->send(200, "application/json", "{\"ok\":true}");
}

// POST handler for /v2hAuto - Toggle V2H auto mode
static void handle_v2hAuto(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->println(" handle_v2hAuto");
    printRequestArgs(debugPtr, request);
  }
  bool currentState = getV2lIsAuto();
  setV2lIsAuto(!currentState);
  fastRefreshNeeded = true;  // next /api/state will return 3 seconds
  request->send(200, "application/json", "{\"ok\":true}");
}

// Helper to escape JSON string values
static String escapeJsonString(const String &str) {
  String escaped = "";
  for (int i = 0; i < str.length(); i++) {
    unsigned char c = str[i];
    switch (c) {
      case '"': escaped += "\\\""; break;
      case '\\': escaped += "\\\\"; break;
      case 10: escaped += "\\n"; break;  // newline (0x0A)
      case 13: escaped += "\\r"; break;  // carriage return (0x0D)
      case 9: escaped += "\\t"; break;   // tab (0x09)
      default:
        if (c >= 32 && c < 127) {
          escaped += (char)c;
        } else {
          // For other control characters, use hex escape
          escaped += "\\u00";
          if (c < 16) escaped += "0";
          char hex[4];
          sprintf(hex, "%x", c);
          escaped += hex;
        }
    }
  }
  return escaped;
}

// GET handler for /api/state - Returns JSON with all dynamic state
static void handle_getState(AsyncWebServerRequest *request) {
  if (debugPtr) {
    debugPtr->println(" handle_getState");
  }

  // Build JSON response with all state variables
  String json = "{";

  // Refresh seconds (3 if fast refresh needed, else 60)
  json += "\"refresh_seconds\":";
  if (isBlanking() || fastRefreshNeeded || (!haveSNTP())) {  // refresh each 1 until get sntp
      fastRefreshNeeded = false;
      refreshSec = 1;
    }
  else {
    refreshSec = DEFAULT_REFRESH;
  }
  json += refreshSec;
  json += ",";

  // 240VAC status
  json += "\"ac240v_status\":\"";
  json += ac240V_active ? "active" : "inactive";
  json += "\",";

  // Charger state and auto mode
  json += "\"charger_state\":\"";
  json += getChargerIsOn() ? "on" : "off";
  json += "\",";
  json += "\"charger_auto_active\":\"";
  json += getChargerIsAuto() ? "active" : "inactive";
  json += "\",";
  json += "\"charger_time_on\":\"";
  json += minToHH_mm(getChargerOnTime());
  json += "\",";
  json += "\"charger_time_off\":\"";
  json += minToHH_mm(getChargerOffTime());
  json += "\",";

  // V2H state and auto mode
  json += "\"v2h_state\":\"";
  json += getV2lIsOn() ? "on" : "off";
  json += "\",";
  json += "\"v2h_auto_active\":\"";
  json += getV2lIsAuto() ? "active" : "inactive";
  json += "\",";
  json += "\"v2h_time_on\":\"";
  json += minToHH_mm(getV2lOnTime());
  json += "\",";
  json += "\"v2h_time_off\":\"";
  json += minToHH_mm(getV2lOffTime());
  json += "\",";

  // Timezone description
  json += "\"tz_desc_str\":\"";
  String TZstr = getTZstr();
  struct posix_tz_data_struct tzdata;
  posixTZDataFromStr(TZstr, tzdata);
  String tzDesc = "";
  buildPOSIXdescription(tzdata, tzDesc);
  json += escapeJsonString(tzDesc);
  json += "\",";

  // Current time and SNTP status
  json += "\"time_current_s\":\"";
  json += String(getLocalTime_s());
  json += "\",";
  json += "\"have_sntp\":";
  json += haveSNTP() ? "true" : "false";
  json += ",";

  // Blanking status - indicates relay switching in progress, user inputs ignored
  json += "\"is_blanking\":";
  json += isBlanking() ? "true" : "false";
  json += "}";

  if (debugPtr) {
    debugPtr->print("Sending state JSON: ");
    debugPtr->println(json);
  }

  AsyncWebServerResponse *response = request->beginResponse(200, "application/json", json);
  response->addHeader("Cache-Control", "no-cache, no-store, must-revalidate");
  response->addHeader("Pragma", "no-cache");
  response->addHeader("Expires", "0");
  request->send(response);
}
