/**
   swtichSettings.cpp
   by Matthew Ford, 
 * (c)2025 Forward Computing and Control Pty. Ltd.
 * NSW Australia, www.forward.com.au
 * This code is not warranted to be fit for any purpose. You may only use it at your own risk.
 * This generated code may be freely used for both private and commercial use
 * provided this copyright is maintained.
*/

#include "switchSettings.h"
#include "LittleFSsupport.h"
#include "ESP32sntpSupport.h"
#include "millisDelay.h"
#include "DebugOut.h"
#include "locks.h"
#include "pins.h"
extern void updateSwitches();

// normally DEBUG is commented out
//#define DEBUG
static Stream* debugPtr = NULL;  // local to this file

millisDelay blankingDelay;
unsigned long BLANKING_DELAY_MS = 10000;  // 10sec between switching from Charging to V2H and visa versa.
// also blanks any other button selections during this time
// Off selections work, including cancel of switchTo... selections
//
static volatile bool blanking = false;
static volatile bool switchConfigChanged = false;  // volatile becaused set by both loop pb's and webpage.cpp threads

unsigned long const PPtoPhaseDelay = 2500;         // delay between switching PP/PE relay and 1Phase relay
static unsigned long CHANGE_OVER_DELAY_MS = 8000;  // ms delay between changing plugs, must be ess than BLANKING_DELAY_MS
static unsigned int DELAY_CYCLES = 100;

static const char SwitchConfigFileName[] = "/switchCfg.bin";  // binary file

struct switchConfig_struct {
  unsigned int charger_onTime;   // in min
  unsigned int charger_offTime;  // in min
  unsigned int v2l_onTime;       // in min
  unsigned int v2l_offTime;      // in min
  bool charger_isAuto;           // true if auto times enabled
  volatile bool charger_isOn;    // true if Charger is ON  volatile for web display  on async thread
  bool v2l_isAuto;               // true if auto times enabled
  volatile bool v2l_isOn;        // true if V2H is ON volatile for web display on async thread
};

static struct switchConfig_struct switchConfig;

// local methods
static struct switchConfig_struct* loadSwitchConfig();
static bool saveSwitchConfig();
static void setupPins();
// void initswitchConfig();
static void setChargerOn(bool value);
static void setV2lOn(bool value);
static void saveSwitchConfigIfNeeded();
static void printSwitchConfig(struct switchConfig_struct& config, Print& out);
static void turnOnCharger(bool turnOn);
static void turnOnV2l(bool turnOn);
static void restartBlanking();

// these methods are protected by a lock
//void setV2lIsOn(bool value); // these just set two volatiles for the loop() to process later, via updateswitchConfig()
//void setChargerIsOn(bool value);  this methods are called by both pushbuttons and webpages.cpp

static uint8_t charger_PP_PE_isOn = LOW;  // HIGH turnS charger PP_PE relay on
static uint8_t v2l_PP_PE_isOn = LOW;      // HIGH turnS v2l PP_PE relay on
static uint8_t charger_1P_isOn = LOW;     // HIGH turnS charger 1Phase relay on
static uint8_t v2l_1P_isOn = LOW;         // HIGH turnS v2l 1Phase relay on

static void ppToPhaseDelay() {
  for (unsigned int i = 0; i < DELAY_CYCLES; i++) {
    delay(PPtoPhaseDelay / DELAY_CYCLES);
    restartBlanking(); // calls  updateSwitches() which resets watchdog
  }
}

static void changeOverDelay() {
  for (unsigned int i = 0; i < DELAY_CYCLES; i++) {
    delay(CHANGE_OVER_DELAY_MS / DELAY_CYCLES);
    restartBlanking(); // calls  updateSwitches() which resets watchdog
  }
}

// ignore other on Request if currently processing one
bool isBlanking() {
  return blanking;
}

// do not set blanking as only set under lock
static void restartBlanking() {
  blankingDelay.start(BLANKING_DELAY_MS);
  updateSwitches();
}

static volatile bool switchToCharging = false;  // set if v2H was on and Charging selected
static volatile bool switchToV2L = false;       // set if Charging was on and V2H selected

static volatile bool chargingFromWeb = false;  // set if v2H was on and Charging selected
static volatile bool webChargingChanged = false;
static volatile bool v2lFromWeb = false;  // set if Charging was on and V2H selected
static volatile bool webV2lChanged = false;

// ========== Relay timings ==================
// PP/PE relay versus 1Phase relay
// Turn On, Close 1Phase Relay, Then PP/PE relay after 500ms
// Turn Off, Open PP/PE relay,  Then 1Phase Relay after 500ms
// for Change over
// Turn Off, Open PP/PE relay,  Then 1Phase Relay after 500ms
// delay 1000ms
// Turn On, Close 1Phase Relay, Then PP/PE relay after 500ms
// total 2sec for change over
// ==========               ==================

// turn on charger turnOn true for on, false for off
// always add restart blanking timer after any change
void turnOnCharger(bool turnOn) {
  restartBlanking();  // reset blanking timer
  if (turnOn) {
    charger_1P_isOn = HIGH;
    digitalWrite(CHARGER_1P_RELAY, charger_1P_isOn);  // HIGH to connect Charger PP/PE to EV car
    ppToPhaseDelay();
    charger_PP_PE_isOn = HIGH;
    digitalWrite(CHARGER_PP_PE_RELAY, charger_PP_PE_isOn);  // HIGH to connect Charger 1 Phase to EV car
    switchConfig.charger_isOn = (charger_PP_PE_isOn == HIGH);
  } else {
    charger_PP_PE_isOn = LOW;
    digitalWrite(CHARGER_PP_PE_RELAY, charger_PP_PE_isOn);  // HIGH to connect Charger 1 Phase to EV car
    ppToPhaseDelay();
    charger_1P_isOn = LOW;
    digitalWrite(CHARGER_1P_RELAY, charger_1P_isOn);  // HIGH to connect Charger PP/PE to EV car
    switchConfig.charger_isOn = (charger_PP_PE_isOn == HIGH);
  }
  restartBlanking();  // reset blanking timer
}

// turn on charger turnOn true for on, false for off
// always add restart blanking timer after any change
void turnOnV2l(bool turnOn) {
  restartBlanking();  // reset blanking timer
  if (turnOn) {
    v2l_1P_isOn = HIGH;
    digitalWrite(V2L_1P_RELAY, v2l_1P_isOn);  // HIGH to connect Charger PP/PE to EV car
    ppToPhaseDelay();
    v2l_PP_PE_isOn = HIGH;
    digitalWrite(V2L_PP_PE_RELAY, v2l_PP_PE_isOn);  // HIGH to connect Charger 1 Phase to EV car
    switchConfig.v2l_isOn = (v2l_PP_PE_isOn == HIGH);
  } else {
    v2l_PP_PE_isOn = LOW;
    digitalWrite(V2L_PP_PE_RELAY, v2l_PP_PE_isOn);  // HIGH to connect Charger 1 Phase to EV car
    ppToPhaseDelay();
    v2l_1P_isOn = LOW;
    digitalWrite(V2L_1P_RELAY, v2l_1P_isOn);  // HIGH to connect Charger PP/PE to EV car
    switchConfig.v2l_isOn = (v2l_PP_PE_isOn == HIGH);
  }
  restartBlanking();  // reset blanking timer
}

// three paths here
// i) turning charger off, open PP/PE contacts, delay(500), open 1Phase contacts
// ii) turning charger on, and v2l off, close 1Phase contacts, delay(500), close PP/PE contacts
// iii) turning charger on, and v2l is on,
//      open V2L PP/PE contacts, delay(500), open V2L 1Phase contacts
//      delay(1000)
//      close charger 1Phase contacts, delay(500), close charger PP/PE contacts

// blocks for upto 3sec
static void setV2lOn(bool on) {
  if (!on) {           // turn off
    turnOnV2l(false);  // turn off sequence
  } else {
    if (charger_1P_isOn == HIGH) {
      turnOnCharger(false);
      changeOverDelay();
    }
    turnOnV2l(true);
  }
}

// blocks for upto 3sec
static void setChargerOn(bool on) {
  if (!on) {               // turn off
    turnOnCharger(false);  // turn off sequence
  } else {
    if (v2l_1P_isOn == HIGH) {
      turnOnV2l(false);
      changeOverDelay();
    }
    turnOnCharger(true);
  }
}

// protected by onOffLock, sets blanking = true; to prevent further changes
void setV2lIsOn(bool value) {
  START_LOCK(onOffLock);
  if (isBlanking()) {  // ignore changes if have one to process
    return;
  }
  blanking = true;
  v2lFromWeb = value;
  webV2lChanged = true;
  END_LOCK(onOffLock);
}

// protected by onOffLock
void setChargerIsOn(bool value) {
  START_LOCK(onOffLock);
  if (isBlanking()) {  // ignore changes if have one to process
    return;
  }
  blanking = true;
  chargingFromWeb = value;
  webChargingChanged = true;
  END_LOCK(onOffLock);
}

// switch on/off at auto times
void handleAuto() {
  static unsigned int lastMin = 0;
  if (isBlanking()) {
    return;  // too soon after another switch
  }
  unsigned int mins = getLocalTime_mins();  // returns (unsigned int)-1 if no ntp
  if (mins >= (24 * 60)) {
    // no ntp times yet
    return;
  }
  if (mins == lastMin) {
    return;  // no new min
  }
  lastMin = mins;
  // check turning ON first as this may force other input off
  if (switchConfig.charger_isAuto && (switchConfig.charger_onTime != switchConfig.charger_offTime)) {
    if (mins == switchConfig.charger_onTime) {
      setChargerOn(true);
      return;  // skip turning v2l OFF as this does this
    }
  }
  if (switchConfig.v2l_isAuto && (switchConfig.v2l_onTime != switchConfig.v2l_offTime)) {
    if (mins == switchConfig.v2l_onTime) {
      setV2lOn(true);
      return;  // skip turning Charger OFF as this does this
    }
  }
  if (switchConfig.charger_isAuto && (switchConfig.charger_onTime != switchConfig.charger_offTime)) {
    if (mins == switchConfig.charger_offTime) {
      setChargerOn(false);
    }
  }
  if (switchConfig.v2l_isAuto && (switchConfig.v2l_onTime != switchConfig.v2l_offTime)) {
    if (mins == switchConfig.v2l_offTime) {
      setV2lOn(false);
    }
  }
}
// handle time delay between turning Charger Off and turning V2L On and visa versa
// no need to use locks here as access to ...Changed vars are atomic and are only set AFTER ..FromWeb are set
void updateSwitchSettings() {
  // pick up volatiles set from web
  if (webV2lChanged) {
    if (v2lFromWeb != getV2lIsOn()) {
      setV2lOn(v2lFromWeb);  // blocks for upto 3sec
    }
    webV2lChanged = false;
  }
  if (webChargingChanged) {
    if (chargingFromWeb != getChargerIsOn()) {
      setChargerOn(chargingFromWeb);  // blocks for upto 3sec
    }
    webChargingChanged = false;
  }

  handleAuto();  // may be blocked by update from web above
  if (blankingDelay.justFinished()) {
    // and change over finished
    switchConfig.charger_isOn = (charger_PP_PE_isOn == HIGH);
    switchConfig.v2l_isOn = (v2l_PP_PE_isOn == HIGH);
    // switchConfigChanged = true; don't update for these as alwasy set to false on restart
    // if (debugPtr) {
    //   debugPtr->println(" clear blanking.");
    // }
    blanking = false;  // volatile and atomic
  }

  saveSwitchConfigIfNeeded();
}

// Getter implementations
int getChargerOnTime() {
  return switchConfig.charger_onTime;
}

int getChargerOffTime() {
  return switchConfig.charger_offTime;
}

int getV2lOnTime() {
  return switchConfig.v2l_onTime;
}

int getV2lOffTime() {
  return switchConfig.v2l_offTime;
}

bool getChargerIsAuto() {
  return switchConfig.charger_isAuto;
}

bool getChargerIsOn() {
  return switchConfig.charger_isOn;
}

bool getV2lIsAuto() {
  return switchConfig.v2l_isAuto;
}

bool getV2lIsOn() {
  return switchConfig.v2l_isOn;
}

void setChargerOnTime(int value) {
  if (switchConfig.charger_onTime != value) {
    switchConfig.charger_onTime = value;
    switchConfigChanged = true;
  }
}

void setChargerOffTime(int value) {
  if (switchConfig.charger_offTime != value) {
    switchConfig.charger_offTime = value;
    switchConfigChanged = true;
  }
}

void setV2lOnTime(int value) {
  if (switchConfig.v2l_onTime != value) {
    switchConfig.v2l_onTime = value;
    switchConfigChanged = true;
  }
}

void setV2lOffTime(int value) {
  if (switchConfig.v2l_offTime != value) {
    switchConfig.v2l_offTime = value;
    switchConfigChanged = true;
  }
}

void setChargerIsAuto(bool value) {
  if (switchConfig.charger_isAuto != value) {
    switchConfig.charger_isAuto = value;
    switchConfigChanged = true;
  }
}


void setV2lIsAuto(bool value) {
  if (switchConfig.v2l_isAuto != value) {
    switchConfig.v2l_isAuto = value;
    switchConfigChanged = true;
  }
}


static void setupPins() {
  // RELAY DRIVES
  pinMode(CHARGER_1P_RELAY, OUTPUT);
  digitalWrite(CHARGER_1P_RELAY, charger_1P_isOn);  // HIGH to connect Charger PP/PE to EV car

  pinMode(V2L_1P_RELAY, OUTPUT);
  digitalWrite(V2L_1P_RELAY, v2l_1P_isOn);  // HIGH to connect V2L PP/PE to EV car

  pinMode(CHARGER_PP_PE_RELAY, OUTPUT);
  digitalWrite(CHARGER_PP_PE_RELAY, charger_PP_PE_isOn);  // HIGH to connect Charger 1 Phase to EV car

  pinMode(V2L_PP_PE_RELAY, OUTPUT);
  digitalWrite(V2L_PP_PE_RELAY, v2l_PP_PE_isOn);  // HIGH to connect V2L 1 Phase to EV car
}

void initSwitchSettings() {
  setupPins();
  initializeLocks();  // for onOffLock
  loadSwitchConfig();
  // on start up ALWAYS start with both OFF!!
  switchConfig.charger_isOn = false;
  switchConfig.v2l_isOn = false;
}

void saveSwitchConfigIfNeeded() {
  if (!switchConfigChanged) {
    return;
  }
  switchConfigChanged = false;
  saveSwitchConfig();  // needs more work
}


// load the last time saved before shutdown/reboot
static bool saveSwitchConfig() {
  if (!initializeFS()) {
    if (debugPtr) {
      debugPtr->println("FS failed to initialize");
    }
    return false;
  }
  // else save config
  File f = LittleFS.open(SwitchConfigFileName, "w");  // create/overwrite
  if (!f) {
    if (debugPtr) {
      debugPtr->print(SwitchConfigFileName);
      debugPtr->print(" did not open for write.");
    }
    return false;  // returns default wrong size
  }
  int bytesOut = f.write((uint8_t*)(&switchConfig), sizeof(struct switchConfig_struct));
  if (bytesOut != sizeof(struct switchConfig_struct)) {
    if (debugPtr) {
      debugPtr->print(SwitchConfigFileName);
      debugPtr->print(" write failed.");
    }
    return false;
  }
  // else return settings
  f.close();  // no rturn
  if (debugPtr) {
    debugPtr->print(SwitchConfigFileName);
    debugPtr->println(" config saved.");
    printSwitchConfig(switchConfig, *debugPtr);
  }
  return true;
}

// used when timeZoneConfigFileName file does not exist or is invalid
void initSwitchConfig() {
  switchConfig.charger_onTime = 0;      // in min
  switchConfig.charger_offTime = 0;     // in min
  switchConfig.v2l_onTime = 0;          // in min
  switchConfig.v2l_offTime = 0;         // in min
  switchConfig.charger_isAuto = false;  // true if auto times enabled
  switchConfig.charger_isOn = false;    // ture if this switch is ON, NOTE: if not selected ALWAYS off
  switchConfig.v2l_isAuto = false;      // true if auto times enabled
  switchConfig.v2l_isOn = false;        // ture if this switch is ON, NOTE: if not selected ALWAYS off
}

// load the last time saved before shutdown/reboot
// returns pointer to timeZoneConfig
static struct switchConfig_struct* loadSwitchConfig() {
#ifdef DEBUG
  debugPtr = getDebugOut();
#endif
  initSwitchConfig();
  if (!initializeFS()) {
    if (debugPtr) {
      debugPtr->println("FS failed to initialize");
    }
    return &switchConfig;  // returns default if cannot open FS
  }
  if (!LittleFS.exists(SwitchConfigFileName)) {
    if (debugPtr) {
      debugPtr->print(SwitchConfigFileName);
      debugPtr->println(" missing.");
    }
    switchConfigChanged = true;
    return &switchConfig;  // returns default if missing
  }
  // else load config
  File f = LittleFS.open(SwitchConfigFileName, "r");
  if (!f) {
    if (debugPtr) {
      debugPtr->print(SwitchConfigFileName);
      debugPtr->print(" did not open for read.");
    }
    LittleFS.remove(SwitchConfigFileName);
    switchConfigChanged = true;
    return &switchConfig;  // returns default wrong size
  }
  if (f.size() != sizeof(switchConfig)) {
    if (debugPtr) {
      debugPtr->print(SwitchConfigFileName);
      debugPtr->print(" wrong size.");
    }
    f.close();
    switchConfigChanged = true;
    return &switchConfig;  // returns default wrong size
  }
  int bytesIn = f.read((uint8_t*)(&switchConfig), sizeof(switchConfig));
  if (bytesIn != sizeof(switchConfig)) {
    if (debugPtr) {
      debugPtr->print(SwitchConfigFileName);
      debugPtr->print(" wrong size read in.");
    }
    initSwitchConfig();  // again
    f.close();
    switchConfigChanged = true;
    return &switchConfig;
  }
  f.close();
  if (debugPtr) {
    debugPtr->println("Loaded switchConfig");
    printSwitchConfig(switchConfig, *debugPtr);
  }
  return &switchConfig;
}

static void printSwitchConfig(struct switchConfig_struct& config, Print& out) {
  out.print("Charger Config  ");
  out.print(config.charger_isOn ? "ON" : "Off");
  out.print(config.charger_isAuto ? "  Auto" : "  notAuto");
  out.print(" onTime:");
  out.print(config.charger_onTime);
  out.print(" offTime:");
  out.println(config.charger_offTime);

  out.print("V2L Config  ");
  out.print(config.v2l_isOn ? "ON" : "Off");
  out.print(config.v2l_isAuto ? "  Auto" : "  notAuto");
  out.print(" onTime:");
  out.print(config.v2l_onTime);
  out.print(" offTime:");
  out.println(config.v2l_offTime);
}
