#ifndef LOCKS_H
#define LOCKS_H

/*
   locks.h
   by Matthew Ford,  2019/11/16
   (c)2019 Forward Computing and Control Pty. Ltd.
   NSW, Australia  www.forward.com.au

*/
#include "freertos/FreeRTOS.h"
#include "freertos/queue.h"
#include "freertos/semphr.h"
#include "finally.h"

/*
volatile unsigned char *my_data;
This declares my_data to be a pointer to volatile unsigned char.

To make the pointer itself volatile, you'd need this instead:
unsigned char *volatile my_data;

And of course, both the pointer and the pointed-to data may be volatile:
volatile unsigned char *volatile my_data;
*/

/*
ESP32 does 32bit atomic read/writes
 debugPtr->println(sizeof(unsigned long)); == 4 i.e. 32bits and so is atomic on ESP32


ESP-IDF FreeRTOS implements Lazy Context Switching for FPUs.
In other words, the state of a core’s FPU registers are not immediately saved when a context switch occurs.
Therefore, tasks that utilize float must be pinned to a particular core upon creation.
If not, ESP-IDF FreeRTOS will automatically pin the task in question to whichever core the task was running on upon the task’s first use of float.
Likewise due to Lazy Context Switching, only interrupt service routines of lowest priority (that is it the Level 1) can use float,
higher priority interrupts do not support FPU usage.
*/

/**
  creates all the locks needed.
  returns non-zero on error<br>
  can be called more the once
  */
bool initializeLocks();

/** a lock control access to on / off volatile variables */
extern SemaphoreHandle_t onOffLock;

/** this macro handles returns from within the lock code with a finally */
#define START_LOCK(lock)  initializeLocks();   (void)xSemaphoreTakeRecursive( lock,portMAX_DELAY); \
    try {\
      finally returnSemaphore([&]{\
        xSemaphoreGiveRecursive( lock );\
        });

/** this macro ends START_LOCK.
 the lock arguement is for documenation only and is not used in the END_LOCK macro*/
#define END_LOCK(lock)     } catch (char *ignore) {  }


#endif
