/**
 * State Management Module
 * Fetches dynamic state from /api/state endpoint and applies to DOM
 * Allows index.html to be fully cached with only state changes transmitted
 */

const StateManager = {
  refreshTimer: null,
  minuteRefreshTimer: null,
  lastMinuteRefreshed: -1,

  // Map of state keys to DOM update handlers
  stateHandlers: {
    'refresh_seconds': (value) => {
      console.log(`StateManager: Setting refresh timer to ${value} seconds`);
      // Clear any existing timer
      if (StateManager.refreshTimer) {
        console.log('StateManager: Clearing existing refresh timer');
        clearTimeout(StateManager.refreshTimer);
      }
      // Set new refresh timer in seconds
      const seconds = parseInt(value);
      if (seconds > 0) {
        console.log(`StateManager: Starting new refresh timer for ${seconds}s`);
        StateManager.refreshTimer = setTimeout(() => {
          console.log('StateManager: Refresh timer fired, fetching state');
          // Re-fetch state instead of reloading page to avoid flashing
          StateManager.fetchAndApplyState();
        }, seconds * 1000);
      }
    },
    'ac240v_status': (value) => {
      const elem = document.getElementById('ac240vStatusBtn');
      if (elem) {
        elem.classList.remove('active', 'inactive');
        elem.classList.add(value === 'active' ? 'active' : 'inactive');
        elem.innerHTML = value === 'active' ? '240VAC<br>ON' : '240VAC<br>OFF';
      }
    },
    'charger_state': (value) => {
      const elem = document.getElementById('chargerOnOffBtn');
      if (elem) {
        elem.classList.remove('on', 'off');
        elem.classList.add(value);
        elem.innerHTML = value === 'on' ? 'Turn Off' : 'Turn On';
      }
    },
    'charger_auto_active': (value) => {
      const btn = document.getElementById('chargerAutoBtn');
      const div = document.getElementById('chargerAutoTimes');
      if (btn) {
        btn.classList.remove('active', 'inactive');
        btn.classList.add(value === 'active' ? 'active' : 'inactive');
      }
      if (div) {
        div.classList.remove('active', 'inactive');
        div.classList.add(value === 'active' ? 'active' : 'inactive');
      }
    },
    'charger_time_on': (value) => {
      const elem = document.querySelector('[data-value="charger-time-on"]');
      if (elem) elem.textContent = value;
    },
    'charger_time_off': (value) => {
      const elem = document.querySelector('[data-value="charger-time-off"]');
      if (elem) elem.textContent = value;
    },
    'v2h_state': (value) => {
      const elem = document.getElementById('v2hOnOffBtn');
      if (elem) {
        elem.classList.remove('on', 'off');
        elem.classList.add(value);
        elem.innerHTML = value === 'on' ? 'Turn Off' : 'Turn On';
      }
    },
    'v2h_auto_active': (value) => {
      const btn = document.getElementById('v2hAutoBtn');
      const div = document.getElementById('v2hAutoTimes');
      if (btn) {
        btn.classList.remove('active', 'inactive');
        btn.classList.add(value === 'active' ? 'active' : 'inactive');
      }
      if (div) {
        div.classList.remove('active', 'inactive');
        div.classList.add(value === 'active' ? 'active' : 'inactive');
      }
    },
    'v2h_time_on': (value) => {
      const elem = document.querySelector('[data-value="v2h-time-on"]');
      if (elem) elem.textContent = value;
    },
    'v2h_time_off': (value) => {
      const elem = document.querySelector('[data-value="v2h-time-off"]');
      if (elem) elem.textContent = value;
    },
    'tz_desc_str': (value) => {
      const elem = document.getElementById('tzDescStr');
      if (elem) {
        // Convert newlines to <br> tags (like processor did)
        const htmlValue = value.replace(/\n/g, '<br>');
        elem.innerHTML = htmlValue;
      }
    },
    'time_current_s': (value) => {
      window.c = parseInt(value) - 1;
    },
    'have_sntp': (value) => {
      window.haveSntp = value === true || value === 1 || value === '1' || value === 'true';
    },
    // Blanking mode indicates relay switching in progress - grey background shows user inputs are ignored
    'is_blanking': (value) => {
      const isBlanking = value === true || value === 1 || value === '1' || value === 'true';
      if (isBlanking) {
        document.body.style.backgroundColor = '#808080'; // grey background during blanking
      } else {
        document.body.style.backgroundColor = ''; // restore default background
      }
    }
  },

  /**
   * Fetch state from server and apply to DOM
   */
  async fetchAndApplyState() {
    try {
      //console.log('StateManager: Fetching /api/state...');
      const response = await fetch('/api/state');
      if (!response.ok) {
        console.error('StateManager: Failed to fetch state:', response.status);
        return;
      }

      const state = await response.json();
      //console.log('StateManager: Received state:', state);
      this.applyState(state);
      //console.log('StateManager: State applied to DOM');
    } catch (error) {
      console.error('StateManager: Error fetching state:', error);
    }
  },

  /**
   * Apply state object to DOM and cache it
   */
  applyState(state) {
    this.applyStateOnly(state);
    // Cache state after applying for use on next page load
    this.cacheState(state);
  },

  /**
   * Apply state to DOM without caching
   */
  applyStateOnly(state) {
    for (const [key, value] of Object.entries(state)) {
      if (this.stateHandlers[key]) {
        //console.log(`StateManager: Applying ${key} = ${value}`);
        this.stateHandlers[key](value);
      } else {
        //console.warn(`StateManager: No handler for ${key}`);
      }
    }
  },

  /**
   * Monitor clock and request refresh 5 seconds after each new minute
   */
  monitorMinuteRefresh() {
    // Only create timer if it doesn't already exist
    if (this.minuteRefreshTimer) {
      return;
    }
    // Check every second if we're at :05 seconds of a new minute
    this.minuteRefreshTimer = setInterval(() => {
      // Get current time in seconds (c is incremented each second)
      const currentSeconds = window.c % 60;
      const currentMinute = Math.floor(window.c / 60);

      // When we hit :05 seconds and haven't refreshed in this minute yet
      if (currentSeconds === 5 && currentMinute !== this.lastMinuteRefreshed) {
        console.log(`StateManager: Triggering minute-based refresh at :05 seconds`);
        this.lastMinuteRefreshed = currentMinute;
        this.fetchAndApplyState();
      }
    }, 1000); // Check every second
  },

  /**
   * Initialize on page load
   */
  init() {
    //console.log('StateManager: init() called, document.readyState =', document.readyState);
    // Wait for DOM to be ready
    if (document.readyState === 'loading') {
      //console.log('StateManager: Waiting for DOMContentLoaded');
      document.addEventListener('DOMContentLoaded', () => this.initialize());
    } else {
      //console.log('StateManager: DOM already ready, initializing');
      this.initialize();
    }
  },

  initialize() {
    //console.log('StateManager: initialize() called');

    // Try to restore last state from localStorage (without re-caching)
    const cachedState = localStorage.getItem('appState');
    if (cachedState) {
      try {
        const state = JSON.parse(cachedState);
        //console.log('StateManager: Restoring cached state');
        this.applyStateOnly(state);  // Apply but don't cache again
      } catch (e) {
        console.warn('StateManager: Failed to parse cached state:', e);
      }
    }

    // Fetch fresh state from server
    this.fetchAndApplyState().then(() => {
      //console.log('StateManager: State fetched, starting clock');
      // Start the clock timer after state is loaded
      if (window.currentTime) {
        window.currentTime();
        // Start monitoring for minute-based refreshes
        this.monitorMinuteRefresh();
      } else {
        console.error('StateManager: window.currentTime not found!');
      }
    });
  },

  /**
   * Save state to localStorage for persistence across page reloads
   */
  cacheState(state) {
    try {
      localStorage.setItem('appState', JSON.stringify(state));
    } catch (e) {
      console.warn('StateManager: Failed to cache state:', e);
    }
  }
};

// Initialize on script load
//console.log('StateManager: Script loaded, calling init()');
StateManager.init();
