/**
   EV_Charge_V2H_switch.ino
   V1.0.1  2026_02_14
   Compiled for DFRobot Beetle ESP32 under ESP32 V3.3.3  (Note: ESP32-C3 and ESP32-C6 did not work reliably)
   For Beetle_ESP32 Select Firebeetle ESP32 for the board
   IO2 (D9) is the builtin led 

   
   by Matthew Ford,  2026/02/14
   (c)2021-2022 Forward Computing and Control Pty. Ltd.
   NSW, Australia  www.forward.com.au
   This code may be freely used for both private and commerical use.
   Provide this copyright is maintained.
*/

/**
Operation.
Set the On/Off times and then select Auto to turn the charger on and off at those times
Selecting On forces the charger on regardless of the time.
Selecting Off forces the charger off regardless of the time.

Push Button Manual override
Use the Manual Push Button to toggle the current state of the charger, On->Off  or Off->On
If the charge is in Auto mode, then this manual state is maintained until the next time change over
That is if the manual state is set to On, then the charger stays on until the next OFF time.
If the manual state is Off, then the charger stays off until the next ON time.
NOTE: If the charger is manually switched OFF in the On period (i.e. between On time and Off time)
the charger will stay off until the next day's On time.

Manual Override from the Web page.
On the web page select On or Off as desired and then if you want to resume Auto operation, select Auto
The current on/off state will be maintained until the next off/on time changes it.

This manual control lets you override/interrup the charging period today and the normal times will 
be used tomorrow.
*/

#include <WiFi.h>
#include "DebugOut.h"
#include "millisDelay.h"
#include "LittleFSsupport.h"
#include "webPages.h"
#include "ESP32sntpSupport.h"
#include "TZ_support.h"
#include "tzPosix.h"
#include "PinFlasher.h"
#include "DebouncedSwitch.h"
#include "switchSettings.h"
#include <ESPAutoWiFiConfig.h>
#include "pins.h"
#include "esp_task_wdt.h"

//#define DEBUG
static Stream* debugPtr = NULL;  // local to this file

bool ac240V_active = false;  // set by Monitor_240VAC accessed by webPages.cpp via extern

static int ledPin = LED_PIN;  // D9 on dfrobot beetle ESP32
static bool highForLedOn = true;  // need to make output high to turn Led ON
static size_t eepromOffset = 0;   // if you use EEPROM.begin(size) in your code add the size here so AutoWiFi data is written after your data

static millisDelay rebootIfOffDelay;
static unsigned long REBOOT_IF_OFF_DELAY_MS = 24ul * 60 * 60 * 1000;  // 24 hrs in ms
static bool rebootWhenOff = false;

// for ESP-01 i.e. Blue led on TX pin, if #define DEBUG is un-commented, led will not flash
// once you have finished debugging comment out #define DEBUG to enable ESP-01 to use GPIO1 (TX) to flash Blue led
// for ESP-01S, Blue led is on GPIO2 and will flash even if debugging is enabled

createBufferedOutput(output, 1024, DROP_IF_FULL);  // buffered out with variable name output with buffer size 512 and mode drop chars if buffer full

/**
  Set a compiled default TZ here. Can be overrided/edited later by webpage.
  get_DefaultTZ() weakly defined in TZ_support.cpp
  if const char *get_DefaultTZ() is not defined then GMT0 is used as the default.
  See 
  https://github.com/nayarsystems/posix_tz_db/blob/master/zones.csv 
  for a list of time zones and their posix_tz strings
  that can be copied and pasted into the get_DefaultTZ or into the Set Time Zone webpage.
*/
const char* get_DefaultTZ() {  // magic name picked up by TZ_Support.cpp
  // TZ_Australia_Sydney
  static char _default_tz_[50] = "AEST-10AEDT,M10.1.0,M4.1.0/3";
  return _default_tz_;
}


// max pin drive (source/HIGH) is 12mA. , max sink/LOW current is 20mA
// Drive capacity current of all GPIO pins total can be 16 x 12 mA.
// https://bbs.espressif.com/viewtopic.php?t=139 and ESP8266 spec for drive current
// NOTE: use 330R resistor to short GPIO0 to GND for programming to prevent shorting out GPIO0 output while being driven high
// then Reset to put into programming mode
// add 1000uF capacitor across pins 1(+ve) and 2(-ve) of the opto-isolator on the relay board to prevent the relay turning on momentarily on power up/reboot

// when using WiFi.config need to manually set the dns servers
// these are the GOOGLE public DNS servers
IPAddress dns1(8, 8, 8, 8);
IPAddress dns2(8, 8, 8, 4);

DebouncedSwitch charger_On_Off(CHARGER_ON_OFF_PB);
DebouncedSwitch v2l_On_Off(V2L_ON_OFF_PB);
DebouncedSwitch Monitor_240VAC(AC_240V_ON);

static void processSwitches();

static void setupPins() {
  // RELAY DRIVES set in switch settings
  initSwitchSettings();

  // PB INPUTS initialised by DebouncedSwitch above
  // 240VAC monitor input from relay contact initialised by DebouncedSwitch above

  // LED DRIVES
  //  AC_MONITOR_LED handled by PinFlasher above
}

void updateSwitches() {
  esp_task_wdt_reset();
  // update push buttons reads
  charger_On_Off.update();
  v2l_On_Off.update();
  Monitor_240VAC.update();
  processSwitches();
}

static void processSwitches() {
  if (v2l_On_Off.isChanged()) {
    if (v2l_On_Off.isDown()) {}
    bool currentState = getV2lIsOn();
    setV2lIsOn(!currentState);  // just set volatiles for later access
  }
  if (charger_On_Off.isChanged()) {
    if (charger_On_Off.isDown()) {}
    bool currentState = getChargerIsOn();
    setChargerIsOn(!currentState);  // just set volatiles for later access
  }
  if (Monitor_240VAC.isDown()) {
    ac240V_active = true;
  } else {
    ac240V_active = false;
  }
}


void setup() {
  setupPins();

#ifdef DEBUG
  Serial.begin(115200);
  for (int i = 8; i > 0; i--) {
    Serial.print(i);
    Serial.println(' ');
    delay(1000);
  }
  Serial.println();
  debugPtr = initializeDebugOut(Serial);  // only need to call this in setup
  // debugPtr = getDebugOut(); // other files call this to get debug stream
  debugPtr->println(" Debug running");
#endif

  initSwitchSettings();
  enableLoopWDT();  // default appears to be 5sec
  rebootIfOffDelay.start(REBOOT_IF_OFF_DELAY_MS);

#ifdef DEBUG
  setESPAutoWiFiConfigDebugOut(Serial);  // turns on debug output for the ESPAutoWiFiConfig code
#endif

  if (ESPAutoWiFiConfigSetup(ledPin, highForLedOn, eepromOffset)) {  // check if we should start access point to configure WiFi settings
    return;                                                          // in config mode so skip rest of setup
  }

#ifdef DEBUG
  // Output MAC address after WiFi is initialized
  uint8_t mac[6];
  WiFi.macAddress(mac);
  if (debugPtr) {
    debugPtr->printf("MAC Address: %02X:%02X:%02X:%02X:%02X:%02X\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
  }
#endif

  if (!initializeFS()) {
    if (debugPtr) {
      debugPtr->println("LittleFS mount failed");
    }
  }
  if (debugPtr) {
    listDir("/", debugPtr);
  }

  initNTP();  // NOTE!!! call this BEFORE calling initTZsupport()
  // because configTime( ) has 0 for the hr and min offset
  initTZsupport();  // starts FS by calling initializeFS()
  startWebServer();
  flushDebugOut();
}

bool isFirst = true;
void loop() {
  updateSwitches();  // reads all pb to allow for manual control if not Wifi

  updateSwitchSettings();  // this handles the volatiles

  pushDebugOut();  // push as much buffereed debug data out as we can, does nothing if debug not initialized
  //  .. other code that MUST run all the time

  if (ESPAutoWiFiConfigLoop()) {  // handle WiFi config webpages
    return;                       // skip the rest of the loop until config finished
  }

  if (saveTZconfigIfNeeded()) {
    //   // saved TZ config changes
  }

  if (rebootIfOffDelay.justFinished()) {
    // reboot every 24hrs if off to clear memory leaks (if any)
    if (debugPtr) {
      debugPtr->println("24hr reboot triggered");
    }
    rebootWhenOff = true;
  }
  if (rebootWhenOff && (!getChargerIsOn()) && (!getV2lIsOn()) && (!isBlanking())) {  // reboot every 24hrs unless something ON
    while (1) {}                                                  // force watch dog timer reboot
  }
  yield();  // yield for other threads to run. e.g. the webpage is running on this cpu with same priority
}
