/*
 * C4001_SafeString_Wrapper.cpp
 * Implementation of wrapper for DFRobot_C4001 sensor
 */

#include "C4001_SafeString_Wrapper.h"
#include <SafeString.h>
#include <pfodDebugPtr.h>

// Global instance
C4001_SafeString_Wrapper C4001_controls;

C4001_SafeString_Wrapper::C4001_SafeString_Wrapper() {
  _sensor = NULL;
  _minDist = 30;
  _maxDist = 2000;
  _trigDist = 600;
}

void C4001_SafeString_Wrapper::setSensor(DFRobot_C4001* sensor) {
  _sensor = sensor;
  if (_sensor) {
    _minDist = _sensor->getMinRange();
    _maxDist = _sensor->getMaxRange();
    _trigDist = _sensor->getTrigRange();
  }
}

const char* C4001_SafeString_Wrapper::getStatusString() {
  cSFA(sfStatusStr, _statusStr);
  sfStatusStr.clear();

  if (_sensor == NULL) {
    sfStatusStr.print("Not initialized");
    return sfStatusStr.c_str();
  }

  sSensorStatus_t status = _sensor->getStatus();

  sfStatusStr.print("Work: ");
  if (status.workStatus == 1) {
    sfStatusStr.print("Running");
  } else {
    sfStatusStr.print("Stopped");
  }

  sfStatusStr.print(" | Mode: ");
  if (status.workMode == 0) {
    sfStatusStr.print("Presence");
  } else {
    sfStatusStr.print("Speed");
  }

  sfStatusStr.print(" | Init: ");
  if (status.initStatus == 1) {
    sfStatusStr.print("OK");
  } else {
    sfStatusStr.print("Failed");
  }

  return sfStatusStr.c_str();
}

const char* C4001_SafeString_Wrapper::getMotionDetectedString() {
  cSFA(sfMotionStr, _motionStr);
  sfMotionStr.clear();

  if (_sensor == NULL) {
    sfMotionStr.print("Error");
    return sfMotionStr.c_str();
  }

  if (_sensor->motionDetection()) {
    sfMotionStr.print("Motion");
  } else {
    sfMotionStr.print("No Motion");
  }

  return sfMotionStr.c_str();
}

int C4001_SafeString_Wrapper::getTrigDist() {
  if (_sensor == NULL) return 600;
  return (int)_sensor->getTrigRange();
}

int C4001_SafeString_Wrapper::getMaxDist() {
  if (_sensor == NULL) return 2000;
  return (int)_sensor->getMaxRange();
}

int C4001_SafeString_Wrapper::getMinDist() {
  if (_sensor == NULL) return 30;
  return (int)_sensor->getMinRange();
}

// Convert from API units (0.01s) to seconds (float)
float C4001_SafeString_Wrapper::getTrigDelay() {
  if (_sensor == NULL) return 0.0;
  uint8_t delayUnits = _sensor->getTrigDelay();
  return (float)delayUnits * 0.01f;
}

// Convert from API units (0.5s) to seconds (float)
float C4001_SafeString_Wrapper::getKeepDelay() {
  if (_sensor == NULL) return 2.0;
  uint16_t delayUnits = _sensor->getKeepTimerout();
  return (float)delayUnits * 0.5f;
}

int C4001_SafeString_Wrapper::getTrigSensitivity() {
  if (_sensor == NULL) return 5;
  return (int)_sensor->getTrigSensitivity();
}

int C4001_SafeString_Wrapper::getKeepSensitivity() {
  if (_sensor == NULL) return 5;
  return (int)_sensor->getKeepSensitivity();
}

void C4001_SafeString_Wrapper::setTrigDist(const byte* inputBytes) {
  if (_sensor == NULL) return;

  char* inputStr = (char*)inputBytes;
  float input = 0;

  // Wrap as SafeString
  cSFP(sfInputStr, inputStr);
  if (!sfInputStr.toFloat(input)) {
    Stream* debugPtr = getDebugPtr();
    if (debugPtr) {
      debugPtr->print("Invalid TrigDist input: '");
      debugPtr->print(inputStr);
      debugPtr->println("'");
    }
    return;
  }

  int dist = (int)input;

  // Limit to valid range 240-2000 cm
  if (dist < 240) dist = 240;
  if (dist > 2000) dist = 2000;

  // Ensure consistency: min <= trig <= max
  if (dist < _minDist) dist = _minDist;
  if (dist > _maxDist) dist = _maxDist;

  _trigDist = dist;
  _sensor->setDetectionRange(_minDist, _maxDist, _trigDist);
}

void C4001_SafeString_Wrapper::setMaxDist(const byte* inputBytes) {
  if (_sensor == NULL) return;

  char* inputStr = (char*)inputBytes;
  float input = 0;

  cSFP(sfInputStr, inputStr);
  if (!sfInputStr.toFloat(input)) {
    Stream* debugPtr = getDebugPtr();
    if (debugPtr) {
      debugPtr->print("Invalid MaxDist input: '");
      debugPtr->print(inputStr);
      debugPtr->println("'");
    }
    return;
  }

  int dist = (int)input;

  // Limit to valid range 240-2000 cm
  if (dist < 240) dist = 240;
  if (dist > 2000) dist = 2000;

  _maxDist = dist;

  // Ensure consistency: min <= trig <= max
  if (_trigDist > _maxDist) _trigDist = _maxDist;
  if (_minDist > _maxDist) _minDist = _maxDist;

  _sensor->setDetectionRange(_minDist, _maxDist, _trigDist);
}

void C4001_SafeString_Wrapper::setMinDist(const byte* inputBytes) {
  if (_sensor == NULL) return;

  char* inputStr = (char*)inputBytes;
  float input = 0;

  cSFP(sfInputStr, inputStr);
  if (!sfInputStr.toFloat(input)) {
    Stream* debugPtr = getDebugPtr();
    if (debugPtr) {
      debugPtr->print("Invalid MinDist input: '");
      debugPtr->print(inputStr);
      debugPtr->println("'");
    }
    return;
  }

  int dist = (int)input;

  // Limit to valid range 30-2000 cm
  if (dist < 30) dist = 30;
  if (dist > 2000) dist = 2000;

  _minDist = dist;

  // Ensure consistency: min <= trig <= max
  if (_trigDist < _minDist) _trigDist = _minDist;
  if (_minDist > _maxDist) _minDist = _maxDist;

  _sensor->setDetectionRange(_minDist, _maxDist, _trigDist);
}

// Convert from seconds (float) to API units (0.01s)
void C4001_SafeString_Wrapper::setTrigDelay(const byte* inputBytes) {
  if (_sensor == NULL) return;

  char* inputStr = (char*)inputBytes;
  float input = 0;

  cSFP(sfInputStr, inputStr);
  if (!sfInputStr.toFloat(input)) {
    Stream* debugPtr = getDebugPtr();
    if (debugPtr) {
      debugPtr->print("Invalid TrigDelay input: '");
      debugPtr->print(inputStr);
      debugPtr->println("'");
    }
    return;
  }

  // Convert from seconds to 0.01s units (0-200)
  uint8_t delayUnits = (uint8_t)(input / 0.01f);

  // Limit to valid range 0-2 seconds (0-200 units)
  if (delayUnits > 200) delayUnits = 200;

  uint16_t keepUnits = _sensor->getKeepTimerout();
  _sensor->setDelay(delayUnits, keepUnits);
}

// Convert from seconds (float) to API units (0.5s)
void C4001_SafeString_Wrapper::setKeepDelay(const byte* inputBytes) {
  if (_sensor == NULL) return;

  char* inputStr = (char*)inputBytes;
  float input = 0;

  cSFP(sfInputStr, inputStr);
  if (!sfInputStr.toFloat(input)) {
    Stream* debugPtr = getDebugPtr();
    if (debugPtr) {
      debugPtr->print("Invalid KeepDelay input: '");
      debugPtr->print(inputStr);
      debugPtr->println("'");
    }
    return;
  }

  // Convert from seconds to 0.5s units (4-3000 range = 2-1500 seconds)
  uint16_t delayUnits = (uint16_t)(input / 0.5f);

  // Limit to valid range 2-1500 seconds (4-3000 units)
  if (delayUnits < 4) delayUnits = 4;
  if (delayUnits > 3000) delayUnits = 3000;

  uint8_t trigUnits = _sensor->getTrigDelay();
  _sensor->setDelay(trigUnits, delayUnits);
}

void C4001_SafeString_Wrapper::setTrigSensitivity(int col) {
  if (_sensor == NULL) return;

  // Limit to valid range 0-9
  uint8_t sensitivity = (uint8_t)(col % 10);

  _sensor->setTrigSensitivity(sensitivity);
}

void C4001_SafeString_Wrapper::setKeepSensitivity(int col) {
  if (_sensor == NULL) return;

  // Limit to valid range 0-9
  uint8_t sensitivity = (uint8_t)(col % 10);

  _sensor->setKeepSensitivity(sensitivity);
}

void C4001_SafeString_Wrapper::buttonStart() {
  if (_sensor == NULL) return;
  _sensor->setSensor(eStartSen);
}

void C4001_SafeString_Wrapper::buttonStop() {
  if (_sensor == NULL) return;
  _sensor->setSensor(eStopSen);
}

void C4001_SafeString_Wrapper::buttonSave() {
  if (_sensor == NULL) return;
  _sensor->setSensor(eSaveParams);
}

void C4001_SafeString_Wrapper::buttonReset() {
  if (_sensor == NULL) return;
  _sensor->setSensor(eResetSen);
}

void C4001_SafeString_Wrapper::buttonRestore() {
  if (_sensor == NULL) return;
  _sensor->setSensor(eRecoverSen);
}
