/**！
 * @file basics.ino
 * @brief This is an example of the C1001 mmWave Human Detection Sensor detecting the presence of people and their respiration and heart rates.
 * 
 * ---------------------------------------------------------------------------------------------------
 *    board   |             MCU                | Leonardo/Mega2560/M0 | ESP32 |  FireBeetle 2 ESP32 V1
 *     VCC    |            3.3V/5V             |        VCC           |  VCC  |  Vcc
 *     GND    |              GND               |        GND           |  GND  |  Gnd
 *     RX     |              TX                |     Serial1 TX1      |  D2   |  D10
 *     TX     |              RX                |     Serial1 RX1      |  D3   |  D11
 * ---------------------------------------------------------------------------------------------------
 * 
 * @copyright  Copyright (c) 2010 DFRobot Co.Ltd (http://www.dfrobot.com)
 * @license     The MIT License (MIT)
 * @author [tangjie](jie.tang@dfrobot.com)
 * @version  V1.0
 * @date  2024-06-03
 * @url https://github.com/DFRobot/DFRobot_HumanDetection
 */

/*   modifications
 * (c)2025 Forward Computing and Control Pty. Ltd.
 * NSW Australia, www.forward.com.au
 * This code is not warranted to be fit for any purpose. You may only use it at your own risk.
 * This generated code may be freely used for both private and commercial use
 * provided this copyright is maintained.
 */

#include "C1001_basics.h"
#include "DFRobot_HumanDetection.h"
#include <pfodDebugPtr.h>
#include <SafeString.h>

#define DEBUG
static Print* debugPtr = NULL;  // local to this file

static bool initialized = false;

static DFRobot_HumanDetection hu(&Serial1);

cSF(sfPresence, 10);
cSF(sfMovement, 10);
cSF(sfMovementRange, 10);
cSF(sfRespiration, 10);
cSF(sfHeartRate, 10);

static int presence = -1;
static int movement = -1;
static int movementRange = -1;
static int respiration = -1;
static int heartRate = -1;

const char *getPresence() {
  sfPresence.clear();
  switch (presence) {
    case -1:
      sfPresence.print("No Data");
      break;
    case 0:
      sfPresence.print("No");
      break;
    case 1:
      sfPresence.print("Yes");
      break;
    default:
      sfPresence.print("No Data");
  }
  return sfPresence.c_str();
}

const char *getMovement() {
  sfMovement.clear();
  switch (movement) {
    case -1:
      sfMovement.print("-");
      break;
    case 1:
      sfMovement.print("Still");
      break;
    case 2:
      sfMovement.print("Active");
      break;
    default:
      sfMovement.print("-");
  }
  return sfMovement.c_str();
}

const char *getMovementRange() {
  sfMovementRange.clear();
  if (movementRange == -1) {
    sfMovementRange.print("-");
  } else {
    sfMovementRange.print(movementRange);
  }
  return sfMovementRange.c_str();
}

const char *getRespiration() {
  sfRespiration.clear();
  if (respiration == -1) {
    sfRespiration.print("-");
  } else {
    sfRespiration.print(respiration);
  }
  return sfRespiration.c_str();
}

const char *getHeartRate() {
  sfHeartRate.clear();
  if (heartRate == -1) {
    sfHeartRate.print("-");
  } else {
    sfHeartRate.print(heartRate);
  }
  return sfHeartRate.c_str();
}


void readingsNotAvailable() {
  movement = -1;
  movementRange = -1;
  respiration = -1;
  heartRate = -1;
}

int C1001_initalize() {
  if (initialized) {
    return 1;  // error to call twice
  }
  initialized = true;
#ifdef DEBUG
  debugPtr = getDebugPtr();
#endif

  presence = -1;
  readingsNotAvailable();

  Serial1.begin(115200, SERIAL_8N1, /*rx =*/D10, /*tx =*/D11);

  if (debugPtr) {
    debugPtr->println("Start initialization");
  }
  while (hu.begin() != 0) {
    if (debugPtr) {
      debugPtr->println("init error!!!");
    }
    return 1;
  }
  if (debugPtr) {
    debugPtr->println("Initialization successful");
  }

  if (debugPtr) {
    debugPtr->println("Start switching work mode");
  }
  if (hu.configWorkMode(hu.eSleepMode) != 0) {
    if (debugPtr) {
      debugPtr->println("error!!!");
    }
    return 1;
  }
  if (debugPtr) {
    debugPtr->println("Work mode switch successful");
  }
  hu.configLEDLight(hu.eHPLed, 1);  // Set HP LED switch, it will not light up even if the sensor detects a person when set to 0.
  hu.sensorRet();                   // Module reset, must perform sensorRet after setting data, otherwise the sensor may not be usable
  return 0;
}

// return 1 on error else 0
int readPresence() {
  if (debugPtr) {
    debugPtr->println("Is Presence: ");
  }
  presence = hu.smHumanData(hu.eHumanPresence);
  switch (presence) {
    case 0:
      if (debugPtr) {
        debugPtr->println("     No one is present");
      }
      readingsNotAvailable();
      return 0;
    case 1:
      if (debugPtr) {
        debugPtr->println("    Someone is present");
      }
      return 0;
    default:
      if (debugPtr) {
        debugPtr->println("    Read error");
      }
      readingsNotAvailable();
      return 1;
  }
}

int readMovement() {
  if (debugPtr) {
    debugPtr->println("Motion: ");
  }
  movement = hu.smHumanData(hu.eHumanMovement);
  switch (movement) {
    case 0:
      if (debugPtr) {
        debugPtr->println("    None");
      }
      return 0;
    case 1:
      if (debugPtr) {
        debugPtr->println("    Still");
      }
      return 0;
    case 2:
      if (debugPtr) {
        debugPtr->println("    Active");
      }
      return 0;
    default:
      if (debugPtr) {
        debugPtr->println("    Read error");
      }
      movement = -1;
      return 0;
  }
}

int readMovementRange() {
  if (debugPtr) {
    debugPtr->println("Movement Range: ");
  }
  movementRange = hu.smHumanData(hu.eHumanMovingRange);
  if (debugPtr) {
    debugPtr->print("    ");
    debugPtr->println(movementRange);
  }
  return 0;
}

int readRespiration() {
  if (debugPtr) {
    debugPtr->println("Respiration rate: ");
  }
  respiration = hu.getBreatheValue();
  if (respiration == 0) {
    respiration = -1;
  }
  if (debugPtr) {
    debugPtr->print("    ");
    debugPtr->println(respiration);
  }
  return 0;
}

int readHeartRate() {
  if (debugPtr) {
    debugPtr->println("Heart rate: ");
  }
  heartRate = hu.getHeartRate();
  if (heartRate == 0) {
    heartRate = -1;
  }
  if (debugPtr) {
    debugPtr->print("    ");
    debugPtr->println(hu.getHeartRate());
  }
  return 0;
}